#!/usr/bin/env php
<?php
/**
 * MetaMiner Automated Intelligence Processing
 * 
 * This script runs MetaMiner intelligence processing on a schedule
 * cPanel Optimized: Uses existing MetaMiner infrastructure
 * 
 * CRON SETUP:
 * Add to cPanel cron jobs:
 * */5 * * * * /usr/bin/php /path/to/core.galliumhash.com/automation/metaminer-processor.php
 */

date_default_timezone_set('UTC');
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Paths
$metaminerPath = __DIR__ . '/../metaminer';
$logFile = __DIR__ . '/logs/metaminer-processor.log';

// Create logs directory
if (!is_dir(dirname($logFile))) {
    mkdir(dirname($logFile), 0755, true);
}

/**
 * Log message with timestamp
 */
function logMessage($message, $level = 'INFO') {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message\n";
    echo "[$timestamp] $message\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

/**
 * Check if processing should run based on data freshness
 */
function shouldRunProcessing($maxAgeMinutes = 10) {
    $lockFile = __DIR__ . '/logs/metaminer-last-run.lock';
    $lastRun = 0;
    
    if (file_exists($lockFile)) {
        $lastRun = (int) file_get_contents($lockFile);
    }
    
    $now = time();
    $timeSinceLastRun = $now - $lastRun;
    $intervalSeconds = $maxAgeMinutes * 60;
    
    return $timeSinceLastRun >= $intervalSeconds;
}

/**
 * Update last run timestamp
 */
function updateLastRun() {
    $lockFile = __DIR__ . '/logs/metaminer-last-run.lock';
    file_put_contents($lockFile, time());
}

/**
 * Run MetaMiner intelligence processing
 */
function runMetaMinerProcessing() {
    global $metaminerPath;
    
    logMessage("Starting MetaMiner intelligence processing...");
    
    try {
        // Change to MetaMiner directory
        chdir($metaminerPath);
        
        // Run intelligence gathering
        $output = [];
        $returnCode = 0;
        exec("php oracle-api.php 2>&1", $output, $returnCode);
        
        $outputStr = implode("\n", $output);
        
        if ($returnCode === 0) {
            logMessage("✅ MetaMiner intelligence processing completed successfully");
            
            // Try to parse response for health status
            $response = json_decode($outputStr, true);
            if ($response && isset($response['data_freshness'])) {
                $freshness = $response['data_freshness'];
                logMessage("Data freshness: " . ($freshness['status'] ?? 'unknown') . 
                          " (" . ($freshness['age_minutes'] ?? '?') . " minutes)");
            }
            
            return [
                'success' => true,
                'output' => $outputStr,
                'timestamp' => date('c')
            ];
        } else {
            logMessage("❌ MetaMiner processing failed with code $returnCode", 'ERROR');
            logMessage("Error: " . substr($outputStr, 0, 500), 'ERROR');
            return [
                'success' => false,
                'error' => $outputStr,
                'return_code' => $returnCode,
                'timestamp' => date('c')
            ];
        }
        
    } catch (Exception $e) {
        logMessage("❌ Exception in MetaMiner processing: " . $e->getMessage(), 'ERROR');
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'timestamp' => date('c')
        ];
    }
}

/**
 * Get MetaMiner health status
 */
function getHealthStatus() {
    try {
        chdir(__DIR__ . '/../metaminer');
        $output = [];
        exec("php oracle-api.php 2>&1", $output, $returnCode);
        
        $response = implode("\n", $output);
        
        // Try to parse JSON response
        $health = json_decode($response, true);
        if ($health) {
            return $health;
        }
        
        return [
            'status' => 'unknown',
            'raw_output' => substr($response, 0, 500),
            'timestamp' => date('c')
        ];
        
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'error' => $e->getMessage(),
            'timestamp' => date('c')
        ];
    }
}

/**
 * Check data freshness from MetaBuilder
 */
function checkDataFreshness() {
    try {
        chdir(__DIR__ . '/../metabuilder');
        $output = [];
        exec("php metabuilder.php --action=data 2>&1", $output, $returnCode);
        
        $response = implode("\n", $output);
        $data = json_decode($response, true);
        
        if ($data) {
            $freshnessStatus = 'unknown';
            $lastCollection = null;
            
            // Try to determine freshness from the response
            if (isset($data['mempool'])) {
                $freshnessStatus = 'fresh';
            } elseif (isset($data['status']) && $data['status'] === 'ok') {
                $freshnessStatus = 'healthy';
            }
            
            return [
                'status' => $freshnessStatus,
                'data_available' => !empty($data),
                'timestamp' => date('c')
            ];
        }
        
        return [
            'status' => 'no_data',
            'timestamp' => date('c')
        ];
        
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'error' => $e->getMessage(),
            'timestamp' => date('c')
        ];
    }
}

// Main execution
try {
    $action = $argv[1] ?? 'run';
    
    switch ($action) {
        case 'run':
            logMessage("MetaMiner Processor starting...");
            
            // Check if we should run based on interval
            if (!shouldRunProcessing(5)) {
                $timeSinceLast = time() - (int) file_get_contents(__DIR__ . '/logs/metaminer-last-run.lock');
                $remaining = (5 * 60) - $timeSinceLast;
                logMessage("Skipping processing - ran recently. " . gmdate("H:i:s", $remaining) . " remaining until next run.");
                exit(0);
            }
            
            // Check data freshness first
            $freshness = checkDataFreshness();
            logMessage("Data freshness check: " . $freshness['status']);
            
            // Run processing
            $result = runMetaMinerProcessing();
            
            // Update last run timestamp
            updateLastRun();
            
            // Log results
            logMessage("Processing result: " . ($result['success'] ? 'SUCCESS' : 'FAILED'));
            
            // Output comprehensive status
            $status = [
                'processing_result' => $result,
                'data_freshness' => $freshness,
                'timestamp' => date('c')
            ];
            
            echo json_encode($status, JSON_PRETTY_PRINT) . "\n";
            
            // Return appropriate exit code
            exit($result['success'] ? 0 : 1);
            
        case 'health':
            $health = getHealthStatus();
            echo json_encode($health, JSON_PRETTY_PRINT) . "\n";
            break;
            
        case 'freshness':
            $freshness = checkDataFreshness();
            echo json_encode($freshness, JSON_PRETTY_PRINT) . "\n";
            break;
            
        case 'test':
            logMessage("MetaMiner Processor - Test mode");
            
            // Test MetaMiner availability
            $health = getHealthStatus();
            echo "MetaMiner Health Check:\n";
            echo "- Status: " . ($health['status'] ?? 'unknown') . "\n";
            
            if (isset($health['data_freshness'])) {
                $freshness = $health['data_freshness'];
                echo "- Data freshness: " . ($freshness['status'] ?? 'unknown') . "\n";
                echo "- Data age: " . ($freshness['age_minutes'] ?? '?') . " minutes\n";
            }
            
            // Check data freshness
            $freshness = checkDataFreshness();
            echo "- Data availability: " . ($freshness['status'] ?? 'unknown') . "\n";
            break;
            
        default:
            echo "Usage: php metaminer-processor.php [run|health|freshness|test]\n";
            echo "  run       - Run intelligence processing\n";
            echo "  health    - Check system health\n";
            echo "  freshness - Check data freshness\n";
            echo "  test      - Test system availability\n";
    }
    
} catch (Exception $e) {
    logMessage("Fatal error: " . $e->getMessage(), 'ERROR');
    echo json_encode([
        'error' => true,
        'message' => $e->getMessage(),
        'timestamp' => date('c')
    ], JSON_PRETTY_PRINT) . "\n";
    exit(1);
}